<?php


namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

//use MercadoPago\PreapprovalPayment;
use MercadoPago\Preapproval;

class MercadoPagoWebhookController extends Controller
{
    public function handle(Request $request)
    {
        Log::info('📥 Webhook recibido:', ['body' => $request->all()]);

        $type = $request->get('type');
        $dataId = $request->input('data.id');

        Log::info('🔍 Data ID extraído:', ['dataId' => $dataId]);

        if ($type === 'payment') {
            try {
                // Verificar que el token de acceso esté configurado
                $accessToken = (config('services.mercadopago.access_token'));
                if (empty($accessToken)) {
                    Log::error('❌ El token de acceso de Mercado Pago no está configurado en .env');
                    return response()->json(['error' => 'Token de acceso no configurado'], 500);
                }

                Log::info('🔧 Token de acceso (parcial):', ['token' => substr($accessToken, 0, 10) . '...']);

                // Hacer una llamada directa a la API de Mercado Pago
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $accessToken,
                    'Accept' => 'application/json'
                ])->get("https://api.mercadopago.com/v1/payments/{$dataId}");

                if ($response->failed()) {
                    Log::warning("⚠️ Error al obtener el pago $dataId: {$response->body()}");
                    return response()->json(['error' => 'No se pudo obtener el pago', 'details' => $response->body()], 400);
                }

                $payment = $response->json();
                Log::info('💳 Detalles del pago:', ['payment' => $payment]);

                if (!isset($payment['status']) || $payment['status'] !== 'approved') {
                    Log::warning("⚠️ Pago $dataId no está aprobado.", ['status' => $payment['status'] ?? 'N/A']);
                    return response()->json(['error' => 'Pago no aprobado'], 400);
                }

                // Obtener external_reference directamente del pago
                $refData = isset($payment['external_reference']) ? json_decode($payment['external_reference'], true) : null;

                if (!$refData) {
                    Log::warning("⚠️ No se encontró external_reference en el pago $dataId.");
                    return response()->json(['error' => 'No se encontró external_reference'], 400);
                }

                Log::info('🔗 Datos de external_reference:', ['refData' => $refData]);

                // Verificar si el pago está asociado con una suscripción
                if (isset($payment['preapproval_id'])) {
                    $preapprovalResponse = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $accessToken,
                        'Accept' => 'application/json'
                    ])->get("https://api.mercadopago.com/preapproval/{$payment['preapproval_id']}");

                    if ($preapprovalResponse->failed()) {
                        Log::warning("⚠️ Suscripción {$payment['preapproval_id']} no encontrada: {$preapprovalResponse->body()}");
                        // Continuar con external_reference del pago si la suscripción no es crítica
                    } else {
                        $preapproval = $preapprovalResponse->json();
                        $refData = json_decode($preapproval['external_reference'], true) ?: $refData;
                        Log::info('🔗 Datos de external_reference (desde suscripción):', ['refData' => $refData]);
                    }
                } else {
                    Log::info("ℹ️ El pago $dataId no está asociado con una suscripción.");
                }

                // Guardar en la base de datos
                DB::table('pagos_cuotas')->insert([
                    'id_pago' => $refData['pagos_id'] ?? null,
                    'numero_cuota' => 1,
                    'fecha_programada' => now(),
                    'fecha_pagada' => now(),
                    'monto' => $payment['transaction_amount'] ?? 0,
                    'estado' => 'pagado',
                    'boleta' => null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

//                DB::table('pedidos')->where('id_pedido', $refData['pedido_id'])->update([
//                    'pedido_activo' => 1
//                ]);


                Log::info("✅ Pago guardado exitosamente: {$payment['id']}");
                return response()->json(['saved' => true]);
            } catch (\Exception $e) {
                Log::error("❌ Error procesando pago: {$e->getMessage()}", [
                    'exception' => $e,
                    'dataId' => $dataId,
                    'trace' => $e->getTraceAsString()
                ]);
                return response()->json(['error' => 'Fallo al procesar el pago', 'details' => $e->getMessage()], 500);
            }
        }

        return response()->json(['message' => 'Evento ignorado']);
    }
}

